/*
 * Decompiled with CFR 0.152.
 */
package unified.bootloader16bit.phy.i2c;

import console16bit.Console16bit;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.logging.Level;
import unified.bootloader16bit.BadResponseException;
import unified.bootloader16bit.Protocol;
import unified.bootloader16bit.ProtocolException;
import unified.bootloader16bit.ProtocolInUseException;
import unified.bootloader16bit.TimeoutException;
import unified.bootloader16bit.phy.Config;
import unified.bootloader16bit.phy.i2c.ConfigImpl;
import unified.bootloader16bit.phy.i2c.Controller;
import unified.bootloader16bit.phy.i2c.DLLWrapper;
import unified.bootloader16bit.phy.i2c.DLLWrapperImpl;
import unified.bootloader16bit.phy.i2c.I2cErrorCode;
import unified.bootloader16bit.phy.i2c.Model;

public class ProtocolI2C
implements Protocol {
    private static final int MCP2221_VID = 1240;
    private static final int MCP2221_PID = 221;
    private static final int I2C_FRAME_HEADER_SIZE = 3;
    private static final int I2C_FRAME_FOOTER_SIZE = 3;
    private static final byte I2C_FRAME_START_MARKER = -91;
    private static final byte I2C_FRAME_END_MARKER = 90;
    private static final byte I2C_FRAME_NOT_READY = 0;
    static final String BAD_CONNECTION_DETAILS = "Unable to establish communication with device.  Common issues are incorrect configuration I2C/Communication settings.  Verify the I2C address is correct and matches the alignment of the firmware.";
    private final ByteBuffer response = ByteBuffer.allocate(1024);
    private final Model model;
    private final Console16bit console;
    private final DLLWrapper mcp2221;
    private final Config<Model, Controller> config;
    private long i2cHandle = -103L;

    public ProtocolI2C(Console16bit console) {
        this(console, new DLLWrapperImpl(), new ConfigImpl(), new Model());
    }

    ProtocolI2C(Console16bit console, DLLWrapper wrapper, Config<Model, Controller> config, Model model) {
        this.console = console;
        this.model = model;
        this.response.order(ByteOrder.LITTLE_ENDIAN);
        this.mcp2221 = wrapper;
        int resultInt = this.mcp2221.loadDll();
        if (resultInt != 0) {
            this.model.setPortSupplier(() -> Collections.emptyList());
            console.addConsoleText(Level.SEVERE, "Unable to load mcp2221_dll_um native DLL: " + resultInt);
        } else {
            this.model.setPortSupplier(this::getPorts);
            console.addConsoleText(Level.FINE, "MCP22221 Library Version: " + this.mcp2221.getLibraryVersion());
        }
        this.config = config;
    }

    @Override
    public void config() {
        try {
            this.config.show(this.model);
        }
        catch (IOException ex) {
            this.console.addConsoleText(Level.SEVERE, "Unable to load configuration settings window.");
        }
    }

    @Override
    public void close() {
        this.mcp2221.close(this.i2cHandle);
    }

    @Override
    public boolean ready() {
        return this.model.isPortSelected();
    }

    private boolean isHandleValid(long handle) {
        return handle >= 0L;
    }

    @Override
    public void open() throws ProtocolInUseException, ProtocolException {
        this.i2cHandle = this.openPort(this.getSelectedPortIndex());
        this.setBitrate(this.i2cHandle, this.model.getBitrate());
    }

    private void setBitrate(long handle, int bitrate) throws ProtocolException {
        int status = this.mcp2221.setSpeed(handle, bitrate);
        if (status != 0) {
            this.mcp2221.close(handle);
            this.console.addConsoleText(Level.FINE, "Error setting speed: " + I2cErrorCode.getErrorName(status));
            throw new ProtocolException("Error opening device: " + I2cErrorCode.getErrorName(status));
        }
    }

    private int getSelectedPortIndex() {
        return this.getPorts().indexOf(this.model.getPort());
    }

    private long openPort(int index) throws ProtocolException, ProtocolInUseException {
        this.console.addConsoleText(Level.FINEST, "Opening device: " + ProtocolI2C.toHexString(1240) + ", " + ProtocolI2C.toHexString(221) + ", " + index);
        if (index < 0) {
            throw new ProtocolException("Attempting to open a device, no devices available");
        }
        long handle = this.mcp2221.openByIndex(1240, 221, index);
        if (!this.isHandleValid(handle)) {
            int lastError = this.mcp2221.getLastError();
            if (lastError == -105) {
                throw new ProtocolInUseException("The requested MCP2221 device is already in use.  Failed to open device.");
            }
            throw new ProtocolException("Error opening device: error code (" + lastError + ")");
        }
        return handle;
    }

    static byte getByte(int in, int position) {
        return (byte)(in >> 8 * position & 0xFF);
    }

    private int i2cFrameSize(int dataPayloadSize) {
        return 3 + dataPayloadSize + 3;
    }

    private I2cErrorCode read(byte[] i2cRxData) {
        return I2cErrorCode.getError(this.mcp2221.i2cRead(this.i2cHandle, i2cRxData.length, this.model.getAddress().byteValue(), true, i2cRxData));
    }

    private I2cErrorCode write(byte[] i2cTxData) {
        return I2cErrorCode.getError(this.mcp2221.i2cWrite(this.i2cHandle, i2cTxData.length, this.model.getAddress().byteValue(), true, i2cTxData));
    }

    private void sendCommand(ByteBuffer commandData) throws IOException {
        ByteBuffer i2cFrame = ByteBuffer.allocate(this.i2cFrameSize(commandData.position()));
        i2cFrame.put((byte)-91);
        i2cFrame.put(ProtocolI2C.getByte(commandData.position(), 0));
        i2cFrame.put(ProtocolI2C.getByte(commandData.position(), 1));
        i2cFrame.put(commandData.array());
        int checksum = this.getChecksum16(commandData.array());
        i2cFrame.put(ProtocolI2C.getByte(checksum, 0));
        i2cFrame.put(ProtocolI2C.getByte(checksum, 1));
        i2cFrame.put((byte)90);
        I2cErrorCode status = this.write(i2cFrame.array());
        if (status.isError()) {
            this.console.addConsoleText(Level.FINE, "Error sending command: " + (Object)((Object)status));
            throw new IOException();
        }
    }

    private int getIntFromShortByteArray(byte[] array) {
        ByteBuffer buffer = ByteBuffer.wrap(array);
        buffer.order(ByteOrder.LITTLE_ENDIAN);
        return buffer.getShort();
    }

    private int getChecksum16(byte[] data) {
        int checksum = 0;
        for (int i = 0; i < data.length; ++i) {
            checksum += Byte.toUnsignedInt(data[i]);
        }
        return checksum & 0xFFFF;
    }

    private void sleep(int duration) {
        try {
            Thread.sleep(duration);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
        }
    }

    private void giveCommandTimeToProcess() {
        this.sleep(10);
    }

    private void waitForStartOfFrame(Integer timeoutMilliseconds) throws IOException, TimeoutException {
        byte[] frameStartIndicator = new byte[1];
        TimeoutThread timeout = new TimeoutThread(timeoutMilliseconds);
        int maxAttempts = 10;
        timeout.start();
        block4: while (timeout.isAlive()) {
            I2cErrorCode status = this.read(frameStartIndicator);
            if (status.isError()) continue;
            switch (frameStartIndicator[0]) {
                case -91: {
                    return;
                }
                case 0: {
                    this.sleep(timeoutMilliseconds / 10);
                    continue block4;
                }
            }
            this.console.addConsoleText(Level.FINE, "Unhandled error waiting for the start of an I2C frame.  Error code = " + (Object)((Object)status));
            throw new IOException("Invalid I2C frame detected.  Invalid start of data frame.");
        }
        throw new TimeoutException();
    }

    private int getDataLength() throws IOException {
        byte[] dataLengthArray = new byte[2];
        I2cErrorCode status = this.read(dataLengthArray);
        if (status.isError()) {
            this.console.addConsoleText(Level.FINE, "Error reading the data length: " + (Object)((Object)status));
            throw new IOException();
        }
        return this.getIntFromShortByteArray(dataLengthArray);
    }

    private byte[] getResponseData(int dataLength) throws IOException {
        byte[] responseBuffer = new byte[dataLength];
        I2cErrorCode status = this.read(responseBuffer);
        if (status.isError()) {
            this.console.addConsoleText(Level.FINE, "Error reading the response data: " + (Object)((Object)status));
            throw new IOException();
        }
        return responseBuffer;
    }

    private void validateFrameIntegrity(byte[] responseData) throws IOException {
        byte[] checksumArray = new byte[2];
        byte[] frameEndIndicator = new byte[1];
        I2cErrorCode status = this.read(checksumArray);
        if (status.isError()) {
            this.console.addConsoleText(Level.FINE, "Error reading the checksum: " + (Object)((Object)status));
            throw new IOException();
        }
        int expectedChecksum = this.getIntFromShortByteArray(checksumArray);
        int calculatedChecksum = this.getChecksum16(responseData);
        status = this.read(frameEndIndicator);
        if (status.isError()) {
            this.console.addConsoleText(Level.FINE, "Error reading the end of I2C frame indicator: " + (Object)((Object)status));
            throw new IOException();
        }
        if (expectedChecksum != calculatedChecksum) {
            throw new IOException("Invalid checksum on frame detected: expected (" + Integer.toHexString(expectedChecksum) + ") read(" + Integer.toHexString(calculatedChecksum));
        }
        if (frameEndIndicator[0] != 90) {
            throw new IOException("Invalid I2C frame detected.  Invalid end of data frame.");
        }
    }

    private ByteBuffer convertResponse(byte[] data) {
        ByteBuffer convertedResponse = ByteBuffer.allocate(data.length);
        convertedResponse.put(data);
        convertedResponse.flip();
        return convertedResponse;
    }

    private ByteBuffer readResponse(Integer timeoutMilliseconds) throws IOException, TimeoutException {
        this.waitForStartOfFrame(timeoutMilliseconds);
        byte[] responseData = this.getResponseData(this.getDataLength());
        this.validateFrameIntegrity(responseData);
        return this.convertResponse(responseData);
    }

    @Override
    public ByteBuffer transceive(ByteBuffer commandData, Integer expectedResponseLength, Integer timeoutMilliseconds) throws IOException, TimeoutException, BadResponseException {
        ByteBuffer trimmedCommand = ByteBuffer.allocate(commandData.position());
        commandData.flip();
        trimmedCommand.put(commandData);
        if (!this.isHandleValid(this.i2cHandle)) {
            throw new IOException("Unable to connect to selected device.");
        }
        this.sendCommand(trimmedCommand);
        this.giveCommandTimeToProcess();
        return this.readResponse(timeoutMilliseconds);
    }

    @Override
    public String getHardwareMisconfigurationMessage() {
        return BAD_CONNECTION_DETAILS;
    }

    ArrayList<String> getPorts() {
        int connectedDevices;
        ArrayList<String> list = new ArrayList<String>();
        if (this.isHandleValid(this.i2cHandle)) {
            this.mcp2221.close(this.i2cHandle);
            this.i2cHandle = -3L;
        }
        if ((connectedDevices = this.mcp2221.getConnectedDevices(1240, 221)) <= 0) {
            this.console.addConsoleText(Level.WARNING, "No connected devices matching VID=" + ProtocolI2C.toHexString(1240) + " and PID=" + ProtocolI2C.toHexString(221) + ". Error Code: " + connectedDevices);
            return list;
        }
        for (int index = 0; index < connectedDevices; ++index) {
            long handle = this.mcp2221.openByIndex(1240, 221, index);
            if (handle == -3L) continue;
            list.add("MCP2221 (" + this.mcp2221.getSerialNumberDescriptor(handle) + ")");
            this.mcp2221.close(handle);
        }
        return list;
    }

    private static String toHexString(int value) {
        return "0x" + Long.toHexString(value).toUpperCase();
    }

    private class TimeoutThread
    extends Thread {
        private Integer duration = 0;

        TimeoutThread(Integer duration) {
            this.duration = duration;
        }

        @Override
        public void run() {
            try {
                TimeoutThread.sleep(this.duration.intValue());
            }
            catch (InterruptedException ex) {
                TimeoutThread.currentThread().interrupt();
            }
        }
    }
}

